<?php
/**
 * Maksupay Payment Gateway.
 *
 * Provides a Maksupay Payment Gateway.
 *
 * @class       Gateway_Maksupay
 * @extends     WC_Payment_Gateway
 * @version     1.0.0
 * @package     WooCommerce\Classes\Payment
 */


if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Gateway_Maksupay Class.
 */
class Gateway_Maksupay extends WC_Payment_Gateway {

	/**
	 * Unique ID for this gateway.
	 *
	 * @var string
	 */
	const ID = 'maksupay';

	/**
	 * Whether or not logging is enabled
	 *
	 * @var bool
	 */
	public static $log_enabled = true;

	/**
	 * Whether or not test local mode is enabled
	 *
	 * @var bool
	 */
	public $test_local_mode = false;

	/**
	 * Logger instance
	 *
	 * @var WC_Logger
	 */
	public static $log = true;

	/**
	 * Whether the debug mode is enabled.
	 *
	 * @var bool
	 */
	public $debug;

	/**
	 * Constructor for the gateway.
	 */
	public function __construct() {
		$this->id                 = self::ID;
		$this->has_fields         = false;
		$this->order_button_text  = __( 'Proceed to Maksupay', 'gateway-maksupay' );
		$this->method_title       = __( 'Maksupay', 'gateway-maksupay' );
		$this->method_description = __( 'Maksupay redirects customers to Maksupay to enter their payment information.', 'gateway-maksupay' );
		$this->supports           = array(
			'products',
			'refunds',
			'woo-payments-blocks',
		);

		// Load the settings.
		$this->init_form_fields();
		$this->init_settings();

		// Define user set variables.
		$this->title           = $this->get_option( 'title' );
		$this->description     = $this->get_option( 'description' );
		$this->debug           = 'yes' === $this->get_option( 'debug', 'no' );
		$this->test_local_mode = 'yes' === $this->get_option( 'test_local_mode', 'no' );
		$this->icon            = plugins_url( 'gateway-maksupay/assets/images/logo.png' );
		self::$log_enabled     = $this->debug;

		add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array(
			$this,
			'process_admin_options'
		) );
	}

	/**
	 * Logging method.
	 *
	 * @param string $message Log message.
	 * @param string $level Optional. Default 'info'. Possible values:
	 *                      emergency|alert|critical|error|warning|notice|info|debug.
	 */
	public static function log( $message, $level = 'info' ) {
		if ( self::$log_enabled ) {
			if ( empty( self::$log ) || is_bool( self::$log ) ) {
				self::$log = wc_get_logger();
			}
			self::$log->log( $level, $message, array( 'source' => self::ID ) );
		}
	}

	/**
	 * Initialise Gateway Settings Form Fields.
	 */
	public function init_form_fields() {
		$this->form_fields = include __DIR__ . '/includes/settings-maksupay.php';
	}

	public function is_available() {
		$is_available = parent::is_available();

		return $is_available && in_array( get_woocommerce_currency(), explode( ', ', $this->get_option( 'available_currencies' ) ) );
	}

	/**
	 * Process the payment and return the result.
	 *
	 * @param int $order_id Order ID.
	 *
	 * @return array
	 */
	public function process_payment( $order_id ) {
		return [
			'result'   => 'success',
			'redirect' => site_url() . '?' . http_build_query( [ 'maksupay_redirect_order_id' => $order_id ] )
		];
	}


	/**
	 * Redirect page with preset values on the form
	 *
	 * @param $order_id
	 *
	 * @return void
	 */
	public function redirect_handler( $order_id ) {
		try {
			require_once __DIR__ . '/includes/class-gateway-maksupay-signature.php';
			$order = wc_get_order( $order_id );

			if ( ! $order ) {
				wp_die( esc_html( __( 'Invalid order', 'gateway-maksupay' ) ) );
			}

			$privateKey = $this->get_option( 'private_key' );
			$url        = $this->get_option( 'redirect_url' );

			$items     = array_values( array_map( function ( WC_Order_Item_Product $item ) use ( $order ) {
				$product     = $item->get_product();
				$quantity    = $item->get_quantity();
				$unit_price  = $order->get_item_total( $item, true ) * 100; // включая налоги, в центах
				$total_price = $unit_price * $quantity;
				$total_tax   = $order->get_item_tax( $item ) * 100;
				$tax_rate    = 0;

				$taxes = $item->get_taxes();
				if ( ! empty( $taxes['total'] ) ) {
					foreach ( $taxes['total'] as $tax_id => $amount ) {
						if ( $amount > 0 ) {
							$tax_obj  = new WC_Tax();
							$rate     = $tax_obj->get_rate_percent( $tax_id );
							$tax_rate = (int) round( $rate * 100 ); // 22% => 2200
							break;
						}
					}
				}

				return [
					't'  => 'p',
					'n'  => $item->get_name(),
					'c'  => $product->get_sku() ?: 'no-sku',
					'q'  => $quantity,
					'qu' => 'pcs',
					'up' => (int) $unit_price,
					'tp' => (int) $total_price,
					'tt' => (int) $total_tax,
					'tr' => (int) $tax_rate,
				];
			}, array_filter( $order->get_items(), fn( $item ) => $item instanceof WC_Order_Item_Product ) ) );
			$orderDesc = 'items:' . json_encode( array_values( $items ), JSON_UNESCAPED_UNICODE );

			$merchantId  = $this->get_option( 'merchant_id' );
			$orderAmount = $order->get_total();
			$currency    = $order->get_currency();

			// Проверка: есть ли shipping-адрес
			$has_shipping = $order->get_shipping_country() || $order->get_shipping_address_1() || $order->get_shipping_city();

			$shipping_country  = null;
			$shipping_state    = null;
			$shipping_postcode = null;
			$shipping_city     = null;
			$shipping_address  = null;
			// Собираем адрес, если есть
			if ( $has_shipping ) {
				$shipping_country  = $order->get_shipping_country();
				$shipping_state    = $order->get_shipping_state();
				$shipping_postcode = $order->get_shipping_postcode();
				$shipping_city     = $order->get_shipping_city();
				$shipping_address  = trim( $order->get_shipping_address_1() . ' ' . $order->get_shipping_address_2() );
			}

			$attempt = $order->get_meta( 'maksupay_payment_attempt' );
			if ( empty( $attempt ) || is_null( $attempt ) ) {
				$attempt = 1;
			} else {
				$attempt = (int) $attempt + 1;
			}

			$order_id = 'WO' . str_pad( $order_id, 9, '0', STR_PAD_LEFT ) . str_pad( $attempt, 3, '0', STR_PAD_LEFT );
			$order->update_meta_data( 'maksupay_payment_order_id', $order_id );
			$order->update_meta_data( 'maksupay_payment_attempt', $attempt );
			$order->save();

			$confirmUrl = $this->get_domain() . '?' . http_build_query( [ 'maksupay_confirm_order_id' => $order_id ] );
			$cancelUrl  = $this->get_domain() . '?' . http_build_query( [ 'maksupay_cancel_order_id' => $order_id ] );

			$payer_name  = $order->get_billing_first_name() . ' ' . $order->get_billing_last_name();
			$payer_email = $order->get_billing_email();
			$payer_phone = $order->get_billing_phone();

			$bill_country = $order->get_billing_country();
			$bill_state   = $order->get_billing_state();
			$bill_zip     = $order->get_billing_postcode();
			$bill_city    = $order->get_billing_city();
			$bill_address = $order->get_billing_address_1() . ' ' . $order->get_billing_address_2();

			$trType = "1";

			$version = "5";

			$signature = ( new Gateway_Maksupay_Signature() )
				->sign_form( $version, $merchantId, $trType, $order_id, $orderDesc, $orderAmount, $currency, $payer_name, $payer_email, $payer_phone, $bill_country, $bill_state, $bill_zip, $bill_city, $bill_address, $shipping_country, $shipping_state, $shipping_postcode, $shipping_city, $shipping_address, $confirmUrl, $cancelUrl, $privateKey );

			?>
            <!DOCTYPE html>
            <html>
            <head>
                <meta charset="UTF-8">
                <title>Starting MaksuPay Payment...</title>
            </head>
            <body>
            <form id="payment-form" action="<?php echo esc_url( $url ) ?>" method="POST">
                <input type="hidden" name="version" value="<?php echo esc_attr( $version ) ?>">
                <input type="hidden" name="mid" value="<?php echo esc_attr( $merchantId ) ?>">
                <input type="hidden" name="trType" value="<?php echo esc_attr( $trType ) ?>">
                <input type="hidden" name="orderid" value="<?php echo esc_attr( $order_id ) ?>">
                <input type="hidden" name="orderDesc" value="<?php echo esc_attr( $orderDesc ) ?>">
                <input type="hidden" name="orderAmount" value="<?php echo esc_attr( $orderAmount ) ?>">
                <input type="hidden" name="currency" value="<?php echo esc_attr( $currency ) ?>">
				<?php if ( ! empty( $payer_name ) ): ?><input type="hidden" name="payerName"
                                                              value="<?php echo esc_attr( $payer_name ) ?>"><?php endif; ?>
				<?php if ( ! empty( $payer_email ) ): ?><input type="hidden" name="payerEmail"
                                                               value="<?php echo esc_attr( $payer_email ) ?>"><?php endif; ?>
				<?php if ( ! empty( $payer_phone ) ): ?><input type="hidden" name="payerPhone"
                                                               value="<?php echo esc_attr( $payer_phone ) ?>"><?php endif; ?>
				<?php if ( ! empty( $bill_country ) ): ?><input type="hidden" name="billCountry"
                                                                value="<?php echo esc_attr( $bill_country ) ?>"><?php endif; ?>
				<?php if ( ! empty( $bill_state ) ): ?><input type="hidden" name="billState"
                                                              value="<?php echo esc_attr( $bill_state ) ?>"><?php endif; ?>
				<?php if ( ! empty( $bill_zip ) ): ?><input type="hidden" name="billZip"
                                                            value="<?php echo esc_attr( $bill_zip ) ?>"><?php endif; ?>
				<?php if ( ! empty( $bill_city ) ): ?><input type="hidden" name="billCity"
                                                             value="<?php echo esc_attr( $bill_city ) ?>"><?php endif; ?>
				<?php if ( ! empty( $bill_address ) ): ?><input type="hidden" name="billAddress"
                                                                value="<?php echo esc_attr( $bill_address ) ?>"><?php endif; ?>
				<?php if ( $has_shipping ): ?>
					<?php if ( ! empty( $shipping_country ) ): ?><input type="hidden" name="shipCountry"
                                                                        value="<?php echo esc_attr( $shipping_country ) ?>"><?php endif; ?>
					<?php if ( ! empty( $shipping_state ) ): ?><input type="hidden" name="shipState"
                                                                      value="<?php echo esc_attr( $shipping_state ) ?>"><?php endif; ?>
					<?php if ( ! empty( $shipping_postcode ) ): ?><input type="hidden" name="shipZip"
                                                                         value="<?php echo esc_attr( $shipping_postcode ) ?>"><?php endif; ?>
					<?php if ( ! empty( $shipping_city ) ): ?><input type="hidden" name="shipCity"
                                                                     value="<?php echo esc_attr( $shipping_city ) ?>"><?php endif; ?>
					<?php if ( ! empty( $shipping_address ) ): ?><input type="hidden" name="shipAddress"
                                                                        value="<?php echo esc_attr( $shipping_address ) ?>"><?php endif; ?>
				<?php endif; ?>
                <input type="hidden" name="confirmUrl" value="<?php echo esc_url( $confirmUrl ) ?>">
                <input type="hidden" name="cancelUrl" value="<?php echo esc_url( $cancelUrl ) ?>">

                <input type="hidden" name="signature" value="<?php echo esc_attr( $signature ) ?>">
            </form>
            <script>
                setTimeout(function () {
                    document.getElementById('payment-form').submit();
                }, 100);
            </script>
            <div style="margin-left: auto; margin-right: auto; text-align: center; margin-top: 20px; magin-bottom: 500px; height: 300px;">
                <h2>Redirecting to MaksuPay to start payment</h2>
                <a href=" / ">Cancel</a>
                <div>
            </html>
			<?php
			wp_die( esc_html( '' ), '', 200 );

		} catch ( \Throwable $e ) {
			self::log( 'Error paymentRequest:' . $e );
			wp_die( esc_html( '' ), '	<h2 style="color: red;">Something went wrong</h2><a href=" / ">Cancel</button></a>', 500 );
		}
		?>
		<?php
	}


	/**
	 * Process a refund if supported.
	 *
	 * @param int $order_id Order ID.
	 * @param float $amount Refund amount.
	 * @param string $reason Refund reason.
	 *
	 * @return bool|WP_Error
	 */
	public function process_refund( $order_id, $amount = null, $reason = '' ) {
		require_once __DIR__ . '/includes/class-gateway-maksupay-signature.php';
		$order             = wc_get_order( $order_id );
		$external_order_id = $order->get_meta( 'maksupay_payment_order_id' );

		$merchantId = $this->get_option( 'merchant_id' );
		$privateKey = $this->get_option( 'private_key' );
		$endpoint   = $this->get_option( 'api_url' );

		$currency = $order->get_currency();

		$attempt = $order->get_meta( 'maksupay_refund_attempt' );
		if ( empty( $attempt ) || is_null( $attempt ) ) {
			$attempt = 1;
		} else {
			$attempt = (int) $attempt + 1;
		}

		$order->update_meta_data( 'maksupay_refund_attempt', $attempt );
		$order->save();

		$orderId         = 'REF-' . $external_order_id . str_pad( $attempt, 3, '0', STR_PAD_LEFT );
		$amountFormatted = number_format( $amount, 2, '.', '' );

		$timestamp = gmdate( "Y - m - d\TH:i:s\Z" );

		$data = [
			"message" => [
				"refundReq" => [
					"merchantId" => $merchantId,
					"orderInfo"  => [
						"orderAmount" => $amountFormatted,
						"currency"    => $currency,
						"orderId"     => $external_order_id
					]
				],
				"id"        => $orderId,
				"version"   => "5.0",
				"ts"        => $timestamp,
				"senderId"  => $merchantId
			]
		];

		$json      = wp_json_encode( $data );
		$signature = 'RSA-SHA256;' . ( new Gateway_Maksupay_Signature() )->sign_raw( $json, $privateKey );

		$certificate = $this->get_option( 'merchant_certificate' );
		$pubKeyHash  = ( new Gateway_Maksupay_Signature() )->get_public_key_hash( $certificate );

		$headers = [
			'X-Payload-Signature' => $signature,
			'X-Sender-Id'         => $merchantId,
			'X-Public-Key-Hash'   => $pubKeyHash,
			'Accept'              => 'application/json',
			'User-Agent'          => 'Modirum HTTPClient',
			'Content-Type'        => 'application/json',
			'Cache-Control'       => 'no-cache',
			'Pragma'              => 'no-cache',
			'Content-Length'      => strlen( $json ),
		];

		self::log( 'Request ' . wp_json_encode( $data ) );
		self::log( 'Headers ' . wp_json_encode( $headers ) );

		$response = wp_remote_post( $endpoint, [
			'headers' => $headers,
			'body'    => $json,
			'timeout' => 30,
		] );

		if ( is_wp_error( $response ) ) {
			self::log( 'VPOS refund failed: ' . $response->get_error_message() );

			return false;
		}

		$http_code = wp_remote_retrieve_response_code( $response );
		$body      = wp_remote_retrieve_body( $response );

		self::log( 'Response ' . $body );

		$responseData = json_decode( $body, true );

		if ( isset( $responseData['message']['errorMessage'] ) || ( isset( $responseData['message']['refundRes']['status'] ) && $responseData['message']['refundRes']['status'] === 'ERROR' ) ) {
			self::log( "VPOS refund failed: " . $responseData['message']['errorMessage'] );

			return false;
		}
		if ( $http_code != 200 ) {
			self::log( "VPOS refund failed: HTTP $http_code" );

			return false;
		}

		return true;
	}

	/**
	 * Capture payment when the order is confirmed
	 *
	 * @param int $order_id Order ID.
	 */
	public function confirm_handler( $order_id ) {
		require_once __DIR__ . '/includes/class-gateway-maksupay-signature.php';
		$order     = wc_get_order( $order_id );
		$post_data = filter_input_array( INPUT_POST, FILTER_DEFAULT );

		self::log( 'Confirm handler: ' . json_encode( $post_data ) );

		$is_signature_valid = ( new Gateway_Maksupay_Signature() )->is_valid_ipn_signature( $post_data, $this->get_option( 'processor_certificate' ), $post_data['signature'] );

		if ( ! $is_signature_valid ) {
			$order->update_status( 'failed', __( 'Invalid signature', 'gateway-maksupay' ) );
			wc_add_notice( __( 'Invalid signature', 'gateway-maksupay' ), 'error' );
			wp_die( esc_html( __( 'Invalid signature', 'gateway-maksupay' ) ), esc_html( __( 'Invalid signature', 'gateway-maksupay' ) ), 500 );

			return;
		}

		$status = $post_data['status'] ?? null;

		switch ( $status ) {
			case 'AUTHORIZED':
			case 'CAPTURED':
				if ( $order->get_status() !== 'processing' && $order->get_status() !== 'completed' ) {
					$order->payment_complete();
					$order->add_order_note( __( 'Payment was successful.', 'gateway-maksupay' ) );
					wc_add_notice( __( 'Payment was successful.', 'gateway-maksupay' ) );
					WC()->cart->empty_cart();
					wp_safe_redirect( get_permalink( wc_get_page_id( 'shop' ) ) );
				}
				break;
			case 'COMPLETED':
				$order->add_order_note( __( 'Action was completed.', 'gateway-maksupay' ) );
				wc_add_notice( __( 'Action was completed.', 'gateway-maksupay' ), 'error' );
				wp_safe_redirect( $order->get_checkout_payment_url() );
				break;
			default:
				$order->update_status( 'failed', __( 'Payment failed', 'gateway-maksupay' ) );
				wc_add_notice( __( 'Payment failed', 'gateway-maksupay' ), 'error' );
				wp_safe_redirect( $order->get_checkout_payment_url() );
				break;
		}
	}

	public function cancel_handler( $order_id ) {
		require_once __DIR__ . '/includes/class-gateway-maksupay-signature.php';
		$post_data = filter_input_array( INPUT_POST, FILTER_DEFAULT );

		$is_signature_valid = ( new Gateway_Maksupay_Signature() )->is_valid_ipn_signature( $post_data, $this->get_option( 'processor_certificate' ), $post_data['signature'] );

		$order = wc_get_order( $order_id );
		if ( ! $is_signature_valid ) {
			$order->update_status( 'failed', __( 'Invalid signature', 'gateway-maksupay' ) );
			wc_add_notice( __( 'Invalid signature', 'gateway-maksupay' ), 'error' );
			wp_die( esc_html( __( 'Invalid signature', 'gateway-maksupay' ) ), esc_html( __( 'Invalid signature', 'gateway-maksupay' ) ), 500 );

			return;
		}

		self::log( 'Cancel handler: ' . json_encode( $post_data ) );

		$status = $post_data['status'] ?? null;

		switch ( $status ) {
			case 'CANCELED':
				$order->update_status( 'cancelled', __( 'Payment canceled.', 'gateway-maksupay' ) );

				wc_add_notice( __( 'Payment canceled.', 'gateway-maksupay' ), 'error' );
				break;

			case 'REFUSED':
			case 'REFUSEDRISK':
			case 'ERROR':
				$order->update_status( 'failed', __( 'Payment failed.', 'gateway-maksupay' ) );

				wc_add_notice( __( 'Payment failed.', 'gateway-maksupay' ), 'error' );
				break;

			default:
				$order->add_order_note( __( 'Payment failed, unexpected status from Gateway.', 'gateway-maksupay' ) );

				wc_add_notice( __( 'Payment failed, unexpected status from Gateway.', 'gateway-maksupay' ), 'error' );
				break;
		}
	}

	protected function get_domain(): string {
		return $this->test_local_mode ? $this->get_option( 'test_local_callback_url' ) : site_url();
	}
}
