/************
 .web.js file
 ************
 Backend '.web.js' files contain functions that run on the server side and can be called from page code.
 ****/

import { Permissions, webMethod } from "wix-web-module";
import { fetch } from 'wix-fetch';
import forge from 'node-forge';
import * as jsdom from 'jsdom';
import { elevate } from "wix-auth";
import { secrets } from "wix-secrets-backend.v2";
import wixSiteBackend from "wix-site-backend";

const ORDERED_KEYS = ['version', 'mid', 'orderid', 'status', 'orderAmount', 'currency', 'paymentTotal', 'message',
    'riskScore', 'payMethod', 'txId', 'paymentRef', 'shipCountry', 'shipState', 'shipZip', 'shipCity', 'shipAddress',
    'shipRecipientName', 'shipRecipientPhone', 'extToken', 'extTokenPanEnd', 'extTokenExp', 'extData', 'var1', 'var2',
    'var3', 'var4', 'var5', 'var6', 'var7', 'var8', 'var9'];
const elevatedGetSecretValue = elevate(secrets.getSecretValue);


export const startPayment = webMethod(
    Permissions.Anyone,
    async ({merchantId, apiVersion, mercCert, procCert, privateKey, serviceUrl, order, domain, wixTransactionId}) => {
        const formData = createFormDataFromOrder(order, merchantId, apiVersion, domain, wixTransactionId);
        const signatureDataString = createPaymentSignatureDataString(formData);
        formData.publicKeyHash = getPublicKeyHashFromCert(mercCert);
        formData.signature = createDigitalSignature(privateKey, signatureDataString);
        const response = await fetch(serviceUrl, {
            headers:{
                'Content-Type': 'application/x-www-form-urlencoded'
            },
            method: 'POST',
            body: new URLSearchParams(formData).toString()
        });
        console.log(response);

        const transactionId = response.url?.split('trid=')[1]?.split('&')[0];
        if (transactionId) {
            return {transactionId: formData.orderid, redirectUrl: response.url};
        }
        const htmlString = await response.text();
        const errorMessage = new jsdom.JSDOM(htmlString).window.document.getElementById('errors')?.textContent || 'Something went wrong';
        throw Error(errorMessage);
    }
);

export const refundPayment = webMethod(
    Permissions.Admin,
    async({refundId, merchantId, apiVersion, mercCert,  procCert, privateKey, orderId, refundAmount, currency, serviceUrl}) => {
        const timeStamp = formatDateToTimestamp(new Date());
        const websitePaymentCurrency = await wixSiteBackend.generalInfo.getPaymentCurrency();
        const body = {"message" : {"refundReq" : {"merchantId" : merchantId, "orderInfo" : {"orderAmount" : refundAmount / getConversionFactorFromIntl(websitePaymentCurrency), "currency" : websitePaymentCurrency, "orderId" : orderId}}, "id" : `M${refundId.split('-').join('')}`, "version" : "5.0", "ts" : timeStamp, "senderId" : merchantId}};
        const publicKeyHash = getPublicKeyHashFromCert(mercCert);
        const signature = createDigitalSignature(privateKey, JSON.stringify(body));


        const response = await fetch(serviceUrl, {
            headers:{
                'Content-Type': 'application/json',
                'X-Sender-ID': merchantId,
                'X-Public-Key-Hash': publicKeyHash,
                'X-Payload-Signature': 'RSA-SHA256;' + signature
            },
            method: 'POST',
            body: JSON.stringify(body)
        });
        console.log(response);

        try {
            return response.buffer().then(async (buffer) => {
                const responseSignatureString = response.headers.get('X-Payload-Signature')?.split('RSA-SHA256;')[1];
                const responsePublicKeyHash = response.headers.get('X-Public-Key-Hash');
                const isResponseValid = await verifyRefundResponse(responseSignatureString, responsePublicKeyHash, buffer);
                if (isResponseValid) {
                    // convert buffer to json and return refund id
                    const jsonString = new TextDecoder('utf-8').decode(buffer);
                    let json = JSON.parse(jsonString);
                    const apiRes = json.message?.refundRes;
                    if (apiRes?.status === 'CAPTURED') {
                        console.log('captured')
                        return {pluginRefundId: apiRes?.orderId};
                    } else {
                        console.log('error', {errorCode: apiRes.errorCode, errorMessage: apiRes.description, reasonCode: 6000, pluginRefundId: apiRes.orderId})
                        return {errorCode: apiRes.errorCode, errorMessage: apiRes.description, reasonCode: 6000, pluginRefundId: apiRes.orderId};
                    }
                } else {
                    throw Error('The refund response from the server could not be verified')
                }
            });
        } catch(error) {
            console.error('Error processing refund:', error?.message);
            throw error;
        }
    });


const createDigitalSignature = (privateKey, data) => {
    const privateKeyObj = forge.pki.privateKeyFromPem(privateKey);
    const md = forge.md.sha256.create();
    md.update(data, 'utf8');
    const signature = privateKeyObj.sign(md);
    return forge.util.encode64(signature);
};

const getPublicKeyHashFromCert = (certificate) => {
    try {
        const cert = forge.pki.certificateFromPem(certificate);
        const publicKey = cert.publicKey;
        const publicKeyAsn1 = forge.pki.publicKeyToAsn1(publicKey);
        const publicKeyDer = forge.asn1.toDer(publicKeyAsn1).getBytes();

        const md = forge.md.sha256.create();
        md.update(publicKeyDer);

        return forge.util.encode64(md.digest().getBytes());
    } catch (error) {
        console.error('Error processing certificate:', error.message);
        throw error;
    }
};

const verifyPaymentResponse = (responseSignatureBase64, responsePublicKeyHashBase64, procCert, responseJson) => {
    let dataString = '';
    ORDERED_KEYS.forEach(key => {
        if (responseJson[key]) {
            dataString += `${responseJson[key]};`
        }
    });
    const md = forge.md.sha256.create();
    md.update(dataString, 'utf8');
    return verifyResponseMessageDigest(responseSignatureBase64, responsePublicKeyHashBase64, md);
};

const verifyRefundResponse = (responseSignatureBase64, responsePublicKeyHashBase64, responseArrayBuffer) => {
    const md = forge.md.sha256.create();
    const arrayBufferBinaryString = String.fromCharCode(...new Uint8Array(responseArrayBuffer))
    md.update(arrayBufferBinaryString, 'binary');
    return verifyResponseMessageDigest(responseSignatureBase64, responsePublicKeyHashBase64, md);
};

const verifyResponseMessageDigest = async (responseSignatureBase64, responsePublicKeyHashBase64, messageDigest) => {
    let procCertsSecret = await elevatedGetSecretValue('maksu-signer-certs');
    let procCertsArray = procCertsSecret.value.split(/(?=-----BEGIN CERTIFICATE-----)/g);

    let procCerts = {};
    procCertsArray.forEach(cert => procCerts[getPublicKeyHashFromCert(cert)] = cert);
    const validatingCert = procCerts[responsePublicKeyHashBase64];

    if (!validatingCert) {
        console.error('Correct validation certificate was not found.');
        return false;
    }

    const signatureBytes = forge.util.decode64(responseSignatureBase64);

    const certificate = forge.pki.certificateFromPem(validatingCert);
    const publicKey = certificate.publicKey;
    const isValidSignature = publicKey.verify(messageDigest.digest().bytes(), signatureBytes);

    if (!isValidSignature) {
        console.error('Invalid signature. Data may have been tampered with.');
        return false;
    }
    return true;
};

const createPaymentSignatureDataString = (formData) => {
    let dataString = '';
    Object.keys(formData).forEach(key => {
        if (formData[key]) {
            dataString += `${formData[key]};`
        }
    });
    return dataString;
};

const formatDateToTimestamp = (date) => {
    return (
        [
            date.getFullYear(),
            padTwoDigits(date.getUTCMonth() + 1),
            padTwoDigits(date.getUTCDate()),
        ].join('-') +
        " " +
        [
            padTwoDigits(date.getUTCHours()),
            padTwoDigits(date.getUTCMinutes()),
            padTwoDigits(date.getUTCSeconds()),
        ].join(":") + '+0000'
    );
    function padTwoDigits(num) {
        return num.toString().padStart(2, '0');
    }
}

const formatOrderItemsForMaksu = (orderItems) => {
    return 'items:' + JSON.stringify(orderItems.map(orderItem => ({
        't': 'p',
        'n': orderItem.name,
        'c': orderItem._id,
        'q': orderItem.quantity,
        'qu': '',
        'up': orderItem.price,
        'tt': 0,
        'tr': 0,
        'tp': orderItem.price * orderItem.quantity
    })));
}

const createFormDataFromOrder = (order, merchantId, apiVersion, domain, wixTransactionId) => {
    return (
        {
            version: `${apiVersion}`,
            mid: `${merchantId}`,
            lang: `${order.description.buyerInfo.buyerLanguage}`,
            orderid: `WX${order._id.split('-').join('').toUpperCase()}`,
            orderDesc: formatOrderItemsForMaksu(order.description.items),
            orderAmount: order.description.totalAmount / 100,
            currency: order.description.currency,
            payerName: `${order.description.shippingAddress.firstName} ${order.description.shippingAddress.lastName}`,
            payerEmail: order.description.shippingAddress.email,
            payerPhone: order.description.shippingAddress.phone,
            billCountry: order.description.billingAddress.countryCode,
            billState: order.description.billingAddress.city,
            billZip: order.description.billingAddress.zipCode,
            billCity: order.description.billingAddress.city,
            billAddress: order.description.billingAddress.address,
            shipState: order.description.shippingAddress.city,
            shipZip: order.description.shippingAddress.zipCode,
            shipCity: order.description.shippingAddress.city,
            shipAddress: order.description.shippingAddress.address,
            // weight: null,
            // dimensions: null,
            confirmUrl: `${domain}/_functions/maksuPayResponseSuccess`,
            cancelUrl: `${domain}/_functions/maksuPayResponseCancel`,
            var1: wixTransactionId
        });
}

function getConversionFactorFromIntl(currencyCode) {
    if (!currencyCode || typeof currencyCode !== 'string') {
        throw Error("Error: Invalid currency code provided.");
    }

    currencyCode = currencyCode.trim().toUpperCase();

    const formatter = new Intl.NumberFormat(undefined, {
        style: 'currency',
        currency: currencyCode
    });

    const { minimumFractionDigits: min, maximumFractionDigits: max } = formatter.resolvedOptions();

    if (typeof min === 'number' && min === max) {
        return Math.pow(10, min);
    } else {
        throw Error(`Error: Ambiguous fraction digits for ${currencyCode} (min: ${min}, max: ${max}).`);
    }
}
