import { response, serverError } from "wix-http-functions";
import { secrets } from "wix-secrets-backend.v2";
import { elevate } from "wix-auth";
import forge from 'node-forge';
import wixPaymentProviderBackend from "wix-payment-provider-backend";


const ORDERED_KEYS = ['version', 'mid', 'orderid', 'status', 'orderAmount', 'currency', 'paymentTotal', 'message',
    'riskScore', 'payMethod', 'txId', 'paymentRef', 'shipCountry', 'shipState', 'shipZip', 'shipCity', 'shipAddress',
    'shipRecipientName', 'shipRecipientPhone', 'extToken', 'extTokenPanEnd', 'extTokenExp', 'extData', 'var1', 'var2',
    'var3', 'var4', 'var5', 'var6', 'var7', 'var8', 'var9'];

const elevatedGetSecretValue = elevate(secrets.getSecretValue);
export function post_maksuPayResponseSuccess(request) {
    const siteUrl = request.url.split('_functions')[0];
    const url = `${siteUrl}payment-success`;
    return request.body.text().then(async (requestBody) => {

        try {
            const jsonRes = urlParamsToJson(requestBody);
            const verified = await verifyPaymentResponse(jsonRes.signature, jsonRes.publicKeyHash, jsonRes);
            if (verified) {
                await wixPaymentProviderBackend.submitEvent({
                    event: {
                        transaction: {
                            wixTransactionId: jsonRes.var1,
                            pluginTransactionId: jsonRes.orderid,
                        },
                    },
                });
                return redirectToUrl(url);
            } else {
                return serverError( {body: { message: "Could not verify the request"}});
            }
        } catch(error) {
            return serverError( {body: { message: "Could not verify the request"}});
        }
    });
}


export async function post_maksuPayResponseCancel(request) {
    const siteUrl = request.url.split('_functions')[0];
    let url = `${siteUrl}payment-failed`;

    try {
        const requestBody = await request.body.text();
        const jsonRes = urlParamsToJson(requestBody);
        if (jsonRes.status === 'CANCELED') {
            url = `${siteUrl}payment-canceled`;
        }
    } catch (error) {
        console.error("Error processing payment cancellation redirect:", error);
        return redirectToUrl(url);
    }

    return redirectToUrl(url);
}

function redirectToUrl(url) {
    return response({
        status: 301,
        headers: {
            location: url
        }});
}

function urlParamsToJson(urlParamsString) {
    const params = new URLSearchParams(urlParamsString);
    const json = {};
    for (const [key, value] of params.entries()) {
        json[key] = value;
    }
    return json;
}

const getPublicKeyHashFromCert = (certificate) => {
    try {
        const cert = forge.pki.certificateFromPem(certificate);
        const publicKey = cert.publicKey;
        const publicKeyAsn1 = forge.pki.publicKeyToAsn1(publicKey);
        const publicKeyDer = forge.asn1.toDer(publicKeyAsn1).getBytes();

        const md = forge.md.sha256.create();
        md.update(publicKeyDer);

        return forge.util.encode64(md.digest().getBytes());
    } catch (error) {
        console.error('Error processing certificate:', error.message);
        throw error;
    }
};


const verifyResponseSignature = async (responseSignatureBase64, responsePublicKeyHashBase64, messageDigest) => {
    let procCertsSecret = await elevatedGetSecretValue('maksu-signer-certs');
    let procCertsArray = procCertsSecret.value.split(/(?=-----BEGIN CERTIFICATE-----)/g);

    let procCerts = {};
    procCertsArray.forEach(cert => procCerts[getPublicKeyHashFromCert(cert)] = cert);
    const validatingCert = procCerts[responsePublicKeyHashBase64];

    if (!validatingCert) {
        console.error('Correct validation certificate was not found.');
        return false;
    }

    const signatureBytes = forge.util.decode64(responseSignatureBase64);

    const certificate = forge.pki.certificateFromPem(validatingCert);
    const publicKey = certificate.publicKey;
    const isValidSignature = publicKey.verify(messageDigest.digest().bytes(), signatureBytes);

    if (!isValidSignature) {
        console.error('Invalid signature. Data may have been tampered with.');
        return false;
    }
    return true;
};

const verifyPaymentResponse = (responseSignatureBase64, responsePublicKeyHashBase64, responseJson) => {
    let dataString = '';
    ORDERED_KEYS.forEach(key => {
        if (responseJson[key]) {
            dataString += `${responseJson[key]};`
        }
    });
    const md = forge.md.sha256.create();
    md.update(dataString, 'utf8');
    return verifyResponseSignature(responseSignatureBase64, responsePublicKeyHashBase64, md);
};
