import * as paymentProvider from 'interfaces-psp-v1-payment-service-provider';
import { startPayment, refundPayment } from 'backend/maksu.web';
import { secrets } from "wix-secrets-backend.v2";
import { elevate } from "wix-auth";
const API_VERSION = 5;
const elevatedGetSecretValue = elevate(secrets.getSecretValue);

/**
 * This payment plugin endpoint is triggered when a merchant provides required data to connect their PSP account to a Wix site.
 * The plugin has to verify merchant's credentials, and ensure the merchant has an operational PSP account.
 * @param {import('interfaces-psp-v1-payment-service-provider').ConnectAccountOptions} options
 * @param {import('interfaces-psp-v1-payment-service-provider').Context} context
 * @returns {Promise<import('interfaces-psp-v1-payment-service-provider').ConnectAccountResponse | import('interfaces-psp-v1-payment-service-provider').BusinessError>}
 */
export const connectAccount = async (options, context) => {
    const {credentials} = options;
    return {
        credentials
    };
};

/**
 * This payment plugin endpoint is triggered when a buyer pays on a Wix site.
 * The plugin has to process this payment request but prevent double payments for the same `wixTransactionId`.
 * @param {import('interfaces-psp-v1-payment-service-provider').CreateTransactionOptions} options
 * @param {import('interfaces-psp-v1-payment-service-provider').Context} context
 * @returns {Promise<import('interfaces-psp-v1-payment-service-provider').CreateTransactionResponse | import('interfaces-psp-v1-payment-service-provider').BusinessError>}
 */
export const createTransaction = async (options, context) => {
    const {merchantCredentials, order} = options;
    console.log(options);
    try {
        const secret = await Promise.all([
            elevatedGetSecretValue('maksu-merchant-cert'),
            elevatedGetSecretValue('maksu-merchant-pri-key')]);


        const response = await startPayment({
            merchantId: merchantCredentials.mid,
            apiVersion: API_VERSION,
            mercCert: secret[0].value,
            privateKey: secret[1].value,
            serviceUrl: merchantCredentials.paymentServiceUrl,
            order: order,
            domain: merchantCredentials.domain,
            wixTransactionId: options.wixTransactionId,
        });
        return {
            pluginTransactionId: response.transactionId,
            redirectUrl: response.redirectUrl
        }
    } catch (error) {
        return {
            errorCode: '', // MaksuPay does not support this at the moment
            errorMessage: error?.message || 'Something went wrong'
        }
    }
};
/**
 * This payment plugin endpoint is triggered when a merchant refunds a payment made on a Wix site.
 * The plugin has to process this refund request but prevent double refunds for the same `wixRefundId`.
 * @param {import('interfaces-psp-v1-payment-service-provider').RefundTransactionOptions} options
 * @param {import('interfaces-psp-v1-payment-service-provider').Context} context
 * @returns {Promise<import('interfaces-psp-v1-payment-service-provider').CreateRefundResponse | import('interfaces-psp-v1-payment-service-provider').BusinessError>}
 */
export const refundTransaction = async (options, context) => {
    const {merchantCredentials, pluginTransactionId, wixRefundId, refundAmount} = options;
    const {currency} = context;
    console.log('refundTransaction');
    console.log(options);
    console.log(context);
    try {
        const secret = await Promise.all([
            elevatedGetSecretValue('maksu-merchant-cert'),
            elevatedGetSecretValue('maksu-signer-certs'),
            elevatedGetSecretValue('maksu-merchant-pri-key')]);


        console.log('refundTransaction2');
        const response = await refundPayment({
            refundId: wixRefundId,
            merchantId: merchantCredentials.mid,
            apiVersion: API_VERSION,
            mercCert:  secret[0].value,
            procCert: secret[1].value,
            privateKey: secret[2].value,
            orderId: pluginTransactionId,
            refundAmount: refundAmount,
            serviceUrl: merchantCredentials.refundServiceUrl
        });
        console.log(response);
        return response;
    } catch(error) {
        return {
            errorCode: '', // MaksuPay does not support this at the moment
            errorMessage: error?.message || 'Something went wrong'
        }
    }
};
