<?php
/**
 * PrestaShop Maksupay Payment Module
 *
 * @author Maksupay Team
 * @copyright Maksupay
 * @license https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */
if (!defined('_PS_VERSION_')) {
    exit;
}

class Maksupay extends PaymentModule
{
    protected $config_form = false;

    public function __construct()
    {
        $this->name = 'maksupay';
        $this->tab = 'payments_gateways';
        $this->version = '1.0.0';
        $this->author = 'Maksupay Team';
        $this->need_instance = 0;

        $this->ps_versions_compliancy = ['min' => '1.6', 'max' => _PS_VERSION_];
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->trans('Maksupay Payment Gateway', [], 'Modules.Maksupay.Admin');
        $this->description = $this->trans(
            'Online Payments You Trust from the Heart of Europe. Accept payments via Maksupay.',
            [],
            'Modules.Maksupay.Admin'
        );

        $this->confirmUninstall = $this->trans(
            'Are you sure you want to uninstall?',
            [],
            'Modules.Maksupay.Admin'
        );

        if (!count(Currency::checkPaymentCurrencies($this->id))) {
            $this->warning = $this->trans(
                'No currency has been set for this module.',
                [],
                'Modules.Maksupay.Admin'
            );
        }

        $this->registerHook('actionOrderSlipAdd');
    }

    /**
     * Don't forget to create update methods if needed:
     * http://doc.prestashop.com/display/PS16/Enabling+the+Auto-Update
     */
    public function install()
    {
        if (extension_loaded('curl') == false) {
            $this->_errors[] = $this->trans(
                'You have to enable the cURL extension on your server to install this module',
                [],
                'Modules.Maksupay.Admin'
            );

            return false;
        }

        return parent::install() && $this->registerHook('paymentOptions') && $this->registerHook('paymentReturn') && $this->registerHook('actionValidateOrder') && $this->createOrderStates();
    }

    public function uninstall()
    {
        Configuration::deleteByName('MAKSUPAY_TEST_LOCAL_MODE');
        Configuration::deleteByName('MAKSUPAY_DEBUG');
        Configuration::deleteByName('MAKSUPAY_MERCHANT_ID');
        Configuration::deleteByName('MAKSUPAY_PRIVATE_KEY');
        Configuration::deleteByName('MAKSUPAY_MERCHANT_CERT');
        Configuration::deleteByName('MAKSUPAY_PROCESSOR_CERT');
        Configuration::deleteByName('MAKSUPAY_REDIRECT_URL');
        Configuration::deleteByName('MAKSUPAY_API_URL');
        Configuration::deleteByName('MAKSUPAY_LOCAL_CALLBACK_URL');

        return parent::uninstall();
    }

    /**
     * Create order states
     */
    protected function createOrderStates()
    {
        $order_states = [
            [
                'name' => 'Awaiting Maksupay payment',
                'color' => '#4169E1',
                'logable' => false,
                'paid' => false,
                'shipped' => false,
                'delivery' => false,
                'hidden' => false,
                'send_email' => false,
                'pdf_invoice' => false,
                'pdf_delivery' => false,
                'module_name' => $this->name,
                'unremovable' => true,
                'template' => '',
            ],
        ];

        foreach ($order_states as $key => $order_state) {
            if (!Configuration::get('MAKSUPAY_OS_' . strtoupper((string) $key))) {
                $OrderState = new OrderState();
                $OrderState->name = [];
                foreach (Language::getLanguages() as $language) {
                    $OrderState->name[$language['id_lang']] = $order_state['name'];
                }
                $OrderState->send_email = $order_state['send_email'];
                $OrderState->color = $order_state['color'];
                $OrderState->hidden = $order_state['hidden'];
                $OrderState->delivery = $order_state['delivery'];
                $OrderState->logable = $order_state['logable'];
                $OrderState->invoice = $order_state['pdf_invoice'];
                $OrderState->module_name = $order_state['module_name'];
                $OrderState->unremovable = $order_state['unremovable'];

                if ($OrderState->add()) {
                    Configuration::updateValue('MAKSUPAY_OS_' . strtoupper((string) $key), (int) $OrderState->id);
                }
            }
        }

        return true;
    }

    /**
     * Load the configuration form
     */
    public function getContent()
    {
        if (Tools::isSubmit('submitMaksupayModule')) {
            $this->postProcess();
        }

        return $this->renderForm();
    }

    /**
     * Create the form that will be displayed in the configuration of your module.
     */
    protected function renderForm()
    {
        $helper = new HelperForm();

        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->default_form_language = $this->context->language->id;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG', 0);

        $helper->identifier = $this->identifier;
        $helper->submit_action = 'submitMaksupayModule';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false)
            . '&configure=' . $this->name . '&tab_module=' . $this->tab . '&module_name=' . $this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');

        $helper->tpl_vars = [
            'fields_value' => $this->getConfigFormValues(),
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        ];

        $html = '';
        if (version_compare(_PS_VERSION_, '1.7.8.0', '<')) {
            $html .= $this->displayWarning(
                $this->trans(
                    'To avoid logout after payment, please configure your server to set cookies with SameSite=None; Secure in .htaccess.',
                    [],
                    'Modules.Maksupay.Admin'
                )
            );
        } else {
            $html .= $this->displayWarning(
                $this->trans(
                    'Make sure "Cookie SameSite" is set to "None" in Advanced Parameters > Administration to avoid logout after external payment.',
                    [],
                    'Modules.Maksupay.Admin'
                )
            );
        }

        $fields = [];
        foreach (['MAKSUPAY_PRIVATE_KEY', 'MAKSUPAY_MERCHANT_CERT', 'MAKSUPAY_PROCESSOR_CERT'] as $field) {
            if (empty(Configuration::get($field))) {
                $fields[] = $field;
            }
        }
        if (!empty($fields)) {
            $html .= $this->displayWarning(
                $this->trans(
                    'Please upload %fields%',
                    ['%fields%' => implode(', ', $fields)],
                    'Modules.Maksupay.Admin'
                )
            );
        }

        $configForm = $this->getConfigForm(
            !empty(Configuration::get('MAKSUPAY_PRIVATE_KEY')),
            !empty(Configuration::get('MAKSUPAY_MERCHANT_CERT')),
            !empty(Configuration::get('MAKSUPAY_PROCESSOR_CERT'))
        );

        return $html . $helper->generateForm([$configForm]);
    }

    /**
     * Create the structure of your form.
     */
    protected function getConfigForm($privateKeyUploaded, $merchantCertificateUploaded, $processorCertificateUploaded)
    {
        return [
            'form' => [
                'legend' => [
                    'title' => $this->trans('Settings', [], 'Modules.Maksupay.Admin'),
                    'icon' => 'icon-cogs',
                ],
                'input' => [
                    [
                        'type' => 'switch',
                        'label' => $this->trans('Test local mode', [], 'Modules.Maksupay.Admin'),
                        'name' => 'MAKSUPAY_TEST_LOCAL_MODE',
                        'is_bool' => true,
                        'desc' => $this->trans('Use this for local testing', [], 'Modules.Maksupay.Admin'),
                        'values' => [
                            [
                                'id' => 'test_on',
                                'value' => true,
                                'label' => $this->trans('Enabled', [], 'Admin.Global'),
                            ],
                            [
                                'id' => 'test_off',
                                'value' => false,
                                'label' => $this->trans('Disabled', [], 'Admin.Global'),
                            ],
                        ],
                    ],
                    [
                        'type' => 'switch',
                        'label' => $this->trans('Debug mode', [], 'Modules.Maksupay.Admin'),
                        'name' => 'MAKSUPAY_DEBUG',
                        'is_bool' => true,
                        'desc' => $this->trans('Enable debug logging', [], 'Modules.Maksupay.Admin'),
                        'values' => [
                            [
                                'id' => 'debug_on',
                                'value' => true,
                                'label' => $this->trans('Enabled', [], 'Admin.Global'),
                            ],
                            [
                                'id' => 'debug_off',
                                'value' => false,
                                'label' => $this->trans('Disabled', [], 'Admin.Global'),
                            ],
                        ],
                    ],
                    [
                        'col' => 3,
                        'type' => 'text',
                        'desc' => $this->trans(
                            'Enter your Maksupay Merchant ID here.',
                            [],
                            'Modules.Maksupay.Admin'
                        ),
                        'name' => 'MAKSUPAY_MERCHANT_ID',
                        'label' => $this->trans('Merchant ID', [], 'Modules.Maksupay.Admin'),
                        'required' => true,
                    ],
                    [
                        'type' => 'file',
                        'label' => $this->trans('Private key', [], 'Modules.Maksupay.Admin'),
                        'name' => 'MAKSUPAY_PRIVATE_KEY',
                        'required' => true,
                        'desc' => $privateKeyUploaded ? $this->trans('Private key uploaded') : $this->trans('Private key was not uploaded.'),
                    ],
                    [
                        'type' => 'file',
                        'label' => $this->trans('Merchant certificate', [], 'Modules.Maksupay.Admin'),
                        'name' => 'MAKSUPAY_MERCHANT_CERT',
                        'desc' => $merchantCertificateUploaded ? $this->trans('Merchant certificate uploaded') : $this->trans('Merchant certificate was not uploaded.'),
                        'required' => true,
                    ],
                    [
                        'type' => 'file',
                        'label' => $this->trans('Processor certificate', [], 'Modules.Maksupay.Admin'),
                        'name' => 'MAKSUPAY_PROCESSOR_CERT',
                        'desc' => $processorCertificateUploaded ? $this->trans('Processor certificate uploaded') : $this->trans('Processor certificate was not uploaded.'),
                        'required' => true,
                    ],
                    [
                        'col' => 6,
                        'type' => 'text',
                        'name' => 'MAKSUPAY_REDIRECT_URL',
                        'label' => $this->trans('Redirect URL', [], 'Modules.Maksupay.Admin'),
                        'desc' => $this->trans('Maksupay gateway redirect URL', [], 'Modules.Maksupay.Admin'),
                        'required' => true,
                    ],
                    [
                        'col' => 6,
                        'type' => 'text',
                        'name' => 'MAKSUPAY_API_URL',
                        'label' => $this->trans('API URL', [], 'Modules.Maksupay.Admin'),
                        'desc' => $this->trans('Maksupay API endpoint URL', [], 'Modules.Maksupay.Admin'),
                        'required' => true,
                    ],
                    [
                        'col' => 6,
                        'type' => 'text',
                        'name' => 'MAKSUPAY_LOCAL_CALLBACK_URL',
                        'label' => $this->trans('Callback URL (local only)', [], 'Modules.Maksupay.Admin'),
                        'desc' => $this->trans(
                            'Used only in test local mode for callbacks',
                            [],
                            'Modules.Maksupay.Admin'
                        ),
                    ],
                ],
                'submit' => [
                    'title' => $this->trans('Save', [], 'Admin.Actions'),
                ],
            ],
        ];
    }

    /**
     * Set values for the inputs.
     */
    protected function getConfigFormValues()
    {
        return [
            'MAKSUPAY_TEST_LOCAL_MODE' => Configuration::get('MAKSUPAY_TEST_LOCAL_MODE'),
            'MAKSUPAY_DEBUG' => Configuration::get('MAKSUPAY_DEBUG'),
            'MAKSUPAY_MERCHANT_ID' => Configuration::get('MAKSUPAY_MERCHANT_ID'),
            'MAKSUPAY_PRIVATE_KEY' => Configuration::get('MAKSUPAY_PRIVATE_KEY'),
            'MAKSUPAY_REDIRECT_URL' => Configuration::get('MAKSUPAY_REDIRECT_URL'),
            'MAKSUPAY_API_URL' => Configuration::get('MAKSUPAY_API_URL'),
            'MAKSUPAY_LOCAL_CALLBACK_URL' => Configuration::get('MAKSUPAY_LOCAL_CALLBACK_URL'),
            'MAKSUPAY_MERCHANT_CERT' => Configuration::get('MAKSUPAY_MERCHANT_CERT'),
            'MAKSUPAY_PROCESSOR_CERT' => Configuration::get('MAKSUPAY_PROCESSOR_CERT'),
        ];
    }

    /**
     * Save form data.
     */
    protected function postProcess()
    {
        $form_values = $this->getConfigFormValues();

        foreach (array_keys($form_values) as $key) {
            if (in_array($key, ['MAKSUPAY_MERCHANT_CERT', 'MAKSUPAY_PRIVATE_KEY', 'MAKSUPAY_PROCESSOR_CERT'])) {
                continue;
            }
            Configuration::updateValue($key, Tools::getValue($key));
        }

        // Handle file uploads
        if (isset($_FILES['MAKSUPAY_MERCHANT_CERT']) && $_FILES['MAKSUPAY_MERCHANT_CERT']['error'] == UPLOAD_ERR_OK) {
            $cert_content = file_get_contents($_FILES['MAKSUPAY_MERCHANT_CERT']['tmp_name']);
            Configuration::updateValue('MAKSUPAY_MERCHANT_CERT', $cert_content);
        }

        if (isset($_FILES['MAKSUPAY_PRIVATE_KEY']) && $_FILES['MAKSUPAY_PRIVATE_KEY']['error'] == UPLOAD_ERR_OK) {
            $cert_content = file_get_contents($_FILES['MAKSUPAY_PRIVATE_KEY']['tmp_name']);
            Configuration::updateValue('MAKSUPAY_PRIVATE_KEY', $cert_content);
        }

        if (isset($_FILES['MAKSUPAY_PROCESSOR_CERT']) && $_FILES['MAKSUPAY_PROCESSOR_CERT']['error'] == UPLOAD_ERR_OK) {
            $cert_content = file_get_contents($_FILES['MAKSUPAY_PROCESSOR_CERT']['tmp_name']);
            Configuration::updateValue('MAKSUPAY_PROCESSOR_CERT', $cert_content);
        }
    }

    /**
     * Return payment options available for PS 1.7+
     */
    public function hookPaymentOptions($params)
    {
        if (!$this->active) {
            return;
        }

        if (!$this->checkCurrency($params['cart'])) {
            return;
        }

        if (Tools::getValue('payment_error') && Tools::getValue('error_message')) {
            $this->context->controller->errors[] = urldecode(Tools::getValue('error_message'));
        }

        $payment_options = [
            $this->getMaksupayPaymentOption(),
        ];

        return $payment_options;
    }

    public function checkCurrency($cart)
    {
        $currency_order = new Currency($cart->id_currency);
        $currencies_module = $this->getCurrency($cart->id_currency);

        if (is_array($currencies_module)) {
            foreach ($currencies_module as $currency_module) {
                if ($currency_order->id == $currency_module['id_currency']) {
                    return true;
                }
            }
        }

        return false;
    }

    public function getMaksupayPaymentOption()
    {
        $maksupayOption = new PrestaShop\PrestaShop\Core\Payment\PaymentOption();
        $maksupayOption->setCallToActionText($this->trans('Pay with Maksupay', [], 'Modules.Maksupay.Shop'))
            ->setAction(
                $this->context->link->getModuleLink(
                    $this->name,
                    'redirect',
                    ['id_shop' => (int) $this->context->shop->id],
                    true
                )
            );

        return $maksupayOption;
    }

    /**
     * Log debug information
     */
    protected function log($message, $level = 1)
    {
        if (Configuration::get('MAKSUPAY_DEBUG')) {
            PrestaShopLogger::addLog(
                '[MAKSUPAY] ' . $message,
                $level,
                null,
                null,
                null,
                true
            );
        }
    }

    /**
     * Process refund
     */
    public function processRefund($order, $amount, $reason = '')
    {
        require_once _PS_MODULE_DIR_ . '/maksupay/classes/signature.php';
        require_once _PS_MODULE_DIR_ . '/maksupay/classes/cookie.php';

        $merchantId = Configuration::get('MAKSUPAY_MERCHANT_ID');
        $privateKey = Configuration::get('MAKSUPAY_PRIVATE_KEY');
        $endpoint = Configuration::get('MAKSUPAY_API_URL');

        $external_order_id = 'PS' . str_pad($order->id, 9, '0', STR_PAD_LEFT) . str_pad(
            '1',
            3,
            '0',
            STR_PAD_LEFT
        ) . $merchantId;
        $currency = new Currency($order->id_currency);

        $orderId = 'REF-' . $external_order_id . '-' . time();
        $amountFormatted = number_format($amount, 2, '.', '');

        $timestamp = gmdate("Y-m-d\TH:i:s\Z");

        $data = [
            'message' => [
                'refundReq' => [
                    'merchantId' => $merchantId,
                    'orderInfo' => [
                        'orderAmount' => $amountFormatted,
                        'currency' => $currency->iso_code,
                        'orderId' => $external_order_id,
                    ],
                ],
                'id' => $orderId,
                'version' => '5.0',
                'ts' => $timestamp,
                'senderId' => $merchantId,
            ],
        ];

        $json = json_encode($data);
        $signature = 'RSA-SHA256;' . (new MaksupaySignature())->signRaw($json, $privateKey);

        $certificate = Configuration::get('MAKSUPAY_MERCHANT_CERT');
        $pubKeyHash = (new MaksupaySignature())->getPublicKeyHash($certificate);

        $headers = [
            'X-Payload-Signature: ' . $signature,
            'X-Sender-Id: ' . $merchantId,
            'X-Public-Key-Hash: ' . $pubKeyHash,
            'Accept: application/json',
            'User-Agent: Modirum HTTPClient',
            'Content-Type: application/json',
            'Cache-Control: no-cache',
            'Pragma: no-cache',
            'Content-Length: ' . strlen($json),
        ];

        $this->log('Refund Request: ' . $json);
        $this->log('Refund Headers: ' . json_encode($headers));

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $endpoint);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $json);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);

        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $this->log('Refund Response: ' . $response);

        if ($response === false) {
            $this->log('CURL error in refund request', 'ERROR');

            return false;
        }

        $responseData = json_decode($response, true);

        if (isset($responseData['message']['errorMessage']) || (isset($responseData['message']['refundRes']['status']) && $responseData['message']['refundRes']['status'] === 'ERROR')) {
            $this->log('Refund failed: ' . $responseData['message']['errorMessage']);

            return false;
        }

        if ($http_code != 200) {
            $this->log("Refund failed: HTTP $http_code");

            return false;
        }

        return true;
    }

    public function hookActionOrderSlipAdd($params)
    {
        /** @var Order $order */
        $order = $params['order'];
        $amount = 0;

        foreach ($params['productList'] as $product) {
            $amount += (float) $product['total_refunded_tax_incl'];
        }

        if (!($order instanceof Order)) {
            $order = new Order((int) $order);
        }
        $amountToRefundShipping = 0.0;

        $allPostValues = Tools::getAllValues();
        if (isset($allPostValues['cancel_product']) && is_array($allPostValues['cancel_product'])) {
            if (isset($allPostValues['cancel_product']['shipping_amount'])) {
                $amountToRefundShipping = (float) $allPostValues['cancel_product']['shipping_amount'];
            }
        }
        $amount += $amountToRefundShipping;

        if (!($order instanceof Order)) {
            $order = new Order((int) $order);
        }
        if ($order->module === $this->name) {
            if ($amount > 0) {
                $success = $this->processRefund($order, $amount);

                if (!$success) {
                    $redirectParams = [
                        'vieworder' => '',
                        'id_order' => $order->id,
                    ];
                    $this->context->container->get('session')->getFlashBag()->add(
                        'error',
                        $this->trans('Maksupay refund failed for order ', [], 'Modules.Maksupay.Admin')
                    );
                    Tools::redirectAdmin((new Link())->getAdminLink('AdminOrders', true, [], $redirectParams));
                }
            } else {
                $this->log(
                    'No amount to refund for order ' . $order->reference . ' on status update, skipping.'
                );
            }
        }
    }

    /**
     * Get domain for callbacks
     */
    protected function getDomain()
    {
        return Configuration::get('MAKSUPAY_TEST_LOCAL_MODE')
            ? Configuration::get('MAKSUPAY_LOCAL_CALLBACK_URL')
            : Tools::getShopDomainSsl(true);
    }
}
