<?php
/**
 * PrestaShop Maksupay Payment Module
 *
 * @author Maksupay Team
 * @copyright Maksupay
 * @license https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */
if (!defined('_PS_VERSION_')) {
    exit;
}

class MaksupayRedirectModuleFrontController extends ModuleFrontController
{
    public $ssl = true;

    /**
     * @see FrontController::initContent()
     */
    public function initContent()
    {
        parent::initContent();

        $cart = $this->context->cart;
        /**
         * @var Maksupay $module
         */
        $module = $this->module;

        if (!$module->checkCurrency($cart)) {
            Tools::redirect('index.php?controller=order');
        }

        $customer = new Customer($cart->id_customer);

        if (!Validate::isLoadedObject($customer)) {
            Tools::redirect('index.php?controller=order&step=1');
        }

        $currency = $this->context->currency;
        $total = (float) $cart->getOrderTotal(true, Cart::BOTH);

        $module->validateOrder(
            $cart->id,
            (int) Configuration::get('MAKSUPAY_OS_0'),
            $total,
            $this->module->displayName,
            null,
            [],
            (int) $currency->id,
            false,
            $customer->secure_key
        );

        $order = new Order($module->currentOrder);

        $this->processPayment($order);
    }

    /**
     * Process payment redirect
     */
    protected function processPayment($order)
    {
        try {
            require_once _PS_MODULE_DIR_ . '/maksupay/classes/signature.php';
            require_once _PS_MODULE_DIR_ . '/maksupay/classes/cookie.php';

            $privateKey = Configuration::get('MAKSUPAY_PRIVATE_KEY');

            $url = Configuration::get('MAKSUPAY_REDIRECT_URL');

            $items = [];
            $cart_products = $order->getProducts();

            foreach ($cart_products as $product) {
                $unit_price = (float) $product['unit_price_tax_incl'] * 100;
                $quantity = (int) $product['product_quantity'];
                $total_price = $unit_price * $quantity;
                $total_tax = ((float) $product['unit_price_tax_incl'] - (float) $product['unit_price_tax_excl']) * $quantity * 100;

                $tax_rate = 0;
                if (isset($product['rate'])) {
                    $tax_rate = (int) round($product['rate'] * 100);
                } elseif (isset($product['tax']) && $product['tax'] > 0 && $product['unit_price_tax_excl'] > 0) {
                    $tax_rate = (int) round($product['tax'] / $product['unit_price_tax_excl'] * 10000);
                }

                $items[] = [
                    't' => 'p',
                    'n' => $product['product_name'],
                    'c' => $product['product_reference'] ?: 'no-sku',
                    'q' => $quantity,
                    'qu' => 'pcs',
                    'up' => (int) $unit_price,
                    'tp' => (int) $total_price,
                    'tt' => (int) $total_tax,
                    'tr' => (int) $tax_rate,
                ];
            }

            // Get customer information
            $customer = new Customer($order->id_customer);
            $address = new Address($order->id_address_invoice);
            $country = new Country($address->id_country);
            $state = new State($address->id_state);

            // Prepare payment parameters
            $version = '5';
            $merchantId = Configuration::get('MAKSUPAY_MERCHANT_ID');
            $trType = '1'; // Purchase
            $orderid = 'PS' . str_pad($order->id, 9, '0', STR_PAD_LEFT) . str_pad(
                '1',
                3,
                '0',
                STR_PAD_LEFT
            ) . $merchantId;
            $orderDesc = 'items:' . json_encode(array_values($items), JSON_UNESCAPED_UNICODE);
            $orderAmount = number_format($order->total_paid, 2, '.', '');
            $currency = (new Currency($order->id_currency))->iso_code;

            $payerName = $customer->firstname . ' ' . $customer->lastname;
            $payerEmail = $customer->email;
            $payerPhone = !empty($address->phone_mobile) ? $address->phone_mobile : $address->phone;

            $billCountry = $country->iso_code;
            $billState = $state->name;
            $billZip = $address->postcode;
            $billCity = $address->city;
            $billAddress = $address->address1 . (!empty($address->address2) ? ', ' . $address->address2 : '');

            // Shipping address (use billing if different shipping not available)
            $ship_address = new Address($order->id_address_delivery);
            $ship_country = new Country($ship_address->id_country);
            $ship_state = new State($ship_address->id_state);

            $shipCountry = $ship_country->iso_code;
            $shipState = $ship_state->name;
            $shipPostcode = $ship_address->postcode;
            $shipCity = $ship_address->city;
            $shipAddress = $ship_address->address1 . (!empty($ship_address->address2) ? ', ' . $ship_address->address2 : '');

            $baseUrl = $this->context->link->getBaseLink();
            $confirmUrl = $this->context->link->getModuleLink(
                'maksupay',
                'confirmation',
                ['order_id' => $order->id],
                true
            );
            $cancelUrl = $this->context->link->getModuleLink(
                'maksupay',
                'cancel',
                ['order_id' => $order->id],
                true
            );

            if (Configuration::get('MAKSUPAY_TEST_LOCAL_MODE')) {
                $localCallbackUrl = Configuration::get('MAKSUPAY_LOCAL_CALLBACK_URL') . '/';

                $confirmUrl = str_replace($baseUrl, $localCallbackUrl, $confirmUrl);
                $cancelUrl = str_replace($baseUrl, $localCallbackUrl, $cancelUrl);
            }

            // Generate signature
            $signature = new MaksupaySignature();
            $signedData = $signature->signForm(
                $version,
                $merchantId,
                $trType,
                $orderid,
                $orderDesc,
                $orderAmount,
                $currency,
                $payerName,
                $payerEmail,
                $payerPhone,
                $billCountry,
                $billState,
                $billZip,
                $billCity,
                $billAddress,
                $shipCountry,
                $shipState,
                $shipPostcode,
                $shipCity,
                $shipAddress,
                $confirmUrl,
                $cancelUrl,
                $privateKey
            );

            // Prepare form data
            $formData = [
                'version' => $version,
                'merchantId' => $merchantId,
                'trType' => $trType,
                'orderid' => $orderid,
                'orderDesc' => $orderDesc,
                'orderAmount' => $orderAmount,
                'currency' => $currency,
                'payerName' => $payerName,
                'payerEmail' => $payerEmail,
                'payerPhone' => $payerPhone,
                'billCountry' => $billCountry,
                'billState' => $billState,
                'billZip' => $billZip,
                'billCity' => $billCity,
                'billAddress' => $billAddress,
                'shipCountry' => $shipCountry,
                'shipState' => $shipState,
                'shipPostcode' => $shipPostcode,
                'shipCity' => $shipCity,
                'shipAddress' => $shipAddress,
                'confirmUrl' => $confirmUrl,
                'cancelUrl' => $cancelUrl,
                'signature' => $signedData,
            ];

            // Assign to template
            $this->context->smarty->assign(array_merge($formData, [
                'url' => $url,
                'has_shipping' => true,
            ]));

            $this->setTemplate('module:maksupay/views/templates/front/redirect.tpl');
        } catch (Exception $e) {
            PrestaShopLogger::addLog(
                'Payment redirect error: ' . $e->getMessage(),
                PrestaShopLogger::LOG_SEVERITY_LEVEL_ERROR
            );
            Tools::redirect('index.php?controller=order&step=1&error=payment_error');
        }
    }
}
