<?php
/**
 * PrestaShop Maksupay Payment Module
 *
 * @author Maksupay Team
 * @copyright Maksupay
 * @license https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 */
if (!defined('_PS_VERSION_')) {
    exit;
}

class MaksupaySignature
{
    /**
     * @param string $version
     * @param string $merchantId
     * @param string $trType
     * @param string $order_id
     * @param string $orderDesc
     * @param string $orderAmount
     * @param string $currency
     * @param string $payerName
     * @param string $payerEmail
     * @param string $payerPhone
     * @param string $billCountry
     * @param string $billState
     * @param string $billZip
     * @param string $billCity
     * @param string $billAddress
     * @param ?string $shippingCountry
     * @param ?string $shippingState
     * @param ?string $shippingPostcode
     * @param ?string $shippingCity
     * @param ?string $shippingAddress
     * @param string $confirmUrl
     * @param string $cancelUrl
     * @param string $privateKey
     *
     * @return string
     */
    public function signForm(
        $version,
        $merchantId,
        $trType,
        $order_id,
        $orderDesc,
        $orderAmount,
        $currency,
        $payerName,
        $payerEmail,
        $payerPhone,
        $billCountry,
        $billState,
        $billZip,
        $billCity,
        $billAddress,
        $shippingCountry,
        $shippingState,
        $shippingPostcode,
        $shippingCity,
        $shippingAddress,
        $confirmUrl,
        $cancelUrl,
        $privateKey
    ) {
        $signatureData = mb_convert_encoding(
            implode(';', array_filter([
                $version,
                $merchantId,
                $trType,
                $order_id,
                $orderDesc,
                $orderAmount,
                $currency,
                $payerName,
                $payerEmail,
                $payerPhone,
                $billCountry,
                $billState,
                $billZip,
                $billCity,
                $billAddress,
                $shippingCountry,
                $shippingState,
                $shippingPostcode,
                $shippingCity,
                $shippingAddress,
                $confirmUrl,
                $cancelUrl,
            ])) . ';',
            'UTF-8'
        );

        openssl_sign($signatureData, $signature, $privateKey, OPENSSL_ALGO_SHA256);

        return base64_encode($signature);
    }

    /**
     * @param array $requestBody
     * @param string $publicKey
     * @param string $requestSignature
     *
     * @return bool
     */
    public function isValidIpnSignature($requestBody, $publicKey, $requestSignature)
    {
        $fieldsOrder = [
            'version',
            'mid',
            'orderid',
            'status',
            'orderAmount',
            'currency',
            'paymentTotal',
            'message',
            'riskScore',
            'payMethod',
            'txId',
            'paymentRef',
            'shipCountry',
            'shipState',
            'shipZip',
            'shipCity',
            'shipAddress',
            'shipRecipientName',
            'shipRecipientPhone',
            'extToken',
            'extTokenPanEnd',
            'extTokenExp',
            'extData',
            'var1',
            'var2',
            'var3',
            'var4',
            'var5',
            'var6',
            'var7',
            'var8',
            'var9',
        ];

        $signatureParts = [];

        foreach ($fieldsOrder as $key) {
            if (isset($requestBody[$key]) && $requestBody[$key] !== '' && $requestBody[$key] !== null and $requestBody[$key] !== 'null') {
                $signatureParts[] = $requestBody[$key];
            }
        }

        $signatureData = mb_convert_encoding(implode(';', $signatureParts) . ';', 'UTF-8');
        $verified = openssl_verify($signatureData, base64_decode($requestSignature), $publicKey, OPENSSL_ALGO_SHA256);

        return $verified === 1;
    }

    public function signRaw(string $body, string $privateKey): string
    {
        openssl_sign(mb_convert_encoding($body, 'UTF-8'), $signature, $privateKey, OPENSSL_ALGO_SHA256);

        return base64_encode($signature);
    }

    public function getPublicKeyHash(string $certificate): string
    {
        $pubKey = openssl_pkey_get_public($certificate);
        if (!$pubKey) {
            throw new Exception('Could not open Public Key');
        }

        $pubKeyDetails = openssl_pkey_get_details($pubKey);
        if (!isset($pubKeyDetails['key'])) {
            throw new Exception('Key does not exist');
        }

        $pubKeyClean = preg_replace('/-----.*?-----|\s/', '', $pubKeyDetails['key']);
        $pubKeyDer = base64_decode($pubKeyClean);
        $hashBin = hash('sha256', $pubKeyDer, true);

        return base64_encode($hashBin);
    }
}
